/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#if MX_OS_WINNT
#include <windows.h>
#include <winsock2.h>
#else
#include <sys/time.h>
#include <unistd.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

#include "mx__lib_types.h"
#include "mcp_config.h"
#include "mx__fops.h"
#include "mx__driver_interface.h"
#include "mx_pin.h"
#include "mx_io.h"

#define MILLION         (1.0e+6)
#define FREE_HACK 0
#define DFLT_BID       0
#define DFLT_EID       0
#define DFLT_ITERATION 1

double
GetTimeInSec(struct timeval tv)
{
        return ((double) (tv.tv_sec) + (double) (tv.tv_usec) * 1e-6);
}

void
usage()
{
  fprintf(stderr, "Usage: mx_reg [args]\n");
  fprintf(stderr, "-b board_id - local Board ID [%d]\n", DFLT_BID);
  fprintf(stderr, "-e local_eid - local endpoint ID [%d]\n", DFLT_EID);
  fprintf(stderr, "-l length - registration length\n");
  fprintf(stderr, "-N iteration - iteration for this length [%d]\n", DFLT_ITERATION);
  fprintf(stderr, "-h - help\n");
}


int main(int argc, char *argv[])
{
  mx_endpt_handle_t fd;
  int i, nsegs, iterations, iter;
  int port, status;
  uint32_t board_id;
  mx_reg_t *reg;
  int size = 5*4096;
  char *buffer;
  uintptr_t va;
  struct timeval start, end;
  double reg_time, dereg_time;
  double reg_sq_time, dereg_sq_time;
  double max_reg_time, min_reg_time;
  double max_dereg_time, min_dereg_time;
  double time;
  double reg_stddev, dereg_stddev;
  mx_set_endpt_t set_endpt;
  int c;

  if (argc < 2) {
    usage();
    exit(1);
  }

  board_id = DFLT_BID;
  port = DFLT_EID;
  iterations = DFLT_ITERATION;
  
  while ((c = getopt(argc, argv, "he:b:l:N:")) != EOF) switch(c) {
  case 'e':
    port = atoi(optarg);
    break;
  case 'b':
    board_id = atoi(optarg);
    break;
  case 'l':
    size = atoi(optarg);
    break;
  case 'N':
    iterations = atoi(optarg);
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  nsegs = /* atoi(argv[2]);*/ 1;

  if (size < 4096) {
    printf("please use a size of 4096 or larger\n");
    exit(1);
  }
  size = (size + MX_RDMA_PAGE_SIZE -1) & ~(MX_RDMA_PAGE_SIZE -1);

  printf("registering %d segments of %d bytes %d times on board %d, endpoint %d\n",
	 nsegs, size, iterations, board_id, port);

  mx_init();
  if (mx__open(board_id, port, &fd) != MX_SUCCESS) {
    printf("open failed\n");
    return 0;
  }

  set_endpt.endpoint = port;
  set_endpt.session_id = 0;
  status = mx__set_endpoint(fd, &set_endpt);
  if (status != 0) {
        perror("ioctl failed");
        exit (1);
    }

  buffer = malloc(size + MX_RDMA_PAGE_SIZE - 1);
  if (!buffer) {
    fprintf(stderr,"Out of memory (%d requested)\n", size);
    exit (1);
  }
  va = ((uintptr_t)buffer + MX_RDMA_PAGE_SIZE -1) & ~(MX_RDMA_PAGE_SIZE -1);
  memset(buffer, 0, size);
  reg = malloc(nsegs * sizeof(*reg));
  if (!reg) {
    perror("malloc failed:\n");
    exit(1);
  }

  reg_time = dereg_time = 0.0;
  min_reg_time = min_dereg_time = 10000.0;
  max_reg_time = max_dereg_time = 0.0;
  reg_sq_time = dereg_sq_time = 0.0;

  for (iter = 0; iter < iterations; iter++) {
    reg->nsegs = nsegs;
    reg->rdma_id = 0;
    reg->memory_context = MX_PIN_LIBMX_CTX;
    if (nsegs > 1) {
      mx_reg_seg_t *segs = malloc(nsegs * sizeof(*segs));
      if (!reg) {
	perror("malloc of segs failed:\n");
	exit(1);
      }
      for (i = 0; i < nsegs; i++) {
	segs[i].vaddr = (uint64_t)va;
	segs[i].len = size;
      }
      reg->segs.vaddr = (uintptr_t)segs;
    } else {
      reg->segs.vaddr = va;
      reg->segs.len = size;
    }
    mx_gettimeofday(&start, NULL);
    status = mx__register(fd, reg);
    mx_gettimeofday(&end, NULL);
    time = GetTimeInSec(end) - GetTimeInSec(start);
    time =  (time / (size / 4096)) * MILLION,
    reg_time += time;
    reg_sq_time += time * time;
    if (time > max_reg_time)
      max_reg_time = time;
    if (time < min_reg_time)
      min_reg_time = time;
    if (status) {
      perror ("mx__register failed:");
      exit(1);
    }


    mx_gettimeofday(&start, NULL);
#if FREE_HACK
    printf("sleeping\n");
    sleep(2);
    printf("freeing\n");
    free(buffer);
    printf("sleeping\n");
    sleep(2);
#endif
    status = mx__deregister(fd, reg->rdma_id);
    mx_gettimeofday(&end, NULL);    
    time = GetTimeInSec(end) - GetTimeInSec(start);
    time =  (time / (size / 4096)) * MILLION,
    dereg_time += time;
    dereg_sq_time += time * time;
    if (time > max_dereg_time)
      max_dereg_time = time;
    if (time < min_dereg_time)
      min_dereg_time = time;

    if (status) {
      perror ("mx__deregister failed\n");
      exit(1);
    }

  }

  reg_time = reg_time / (double)iterations;
  printf("registration (mean/max/min us/page): \t%2.2f / %2.2f / %2.2f",
	 reg_time, max_reg_time, min_reg_time);
  if (iterations > 1) {
    reg_stddev = sqrt((reg_sq_time - (double)iterations * reg_time * reg_time)
		      / ((double) iterations - 1.0));
    printf("\tstddev=%2.4f\n", reg_stddev);
  } else {
    printf("\n");
  }

  dereg_time = dereg_time / (double)iterations;
  printf("deregistration (mean/max/min us/page): \t%2.2f / %2.2f / %2.2f",
	 dereg_time, max_dereg_time, min_dereg_time);


  if (iterations > 1) {
    dereg_stddev = sqrt((dereg_sq_time - (double)iterations * dereg_time * dereg_time)
		      / ((double) iterations - 1.0));
    printf("\tstddev=%2.4f\n", dereg_stddev);
  }
  else {
    printf("\n");
  }

  if (status) {
    perror ("mx__register failed\n");
    exit(1);
  }
  fprintf(stderr, "it worked\n");
  mx__close(fd);

  return 0;
}
